import org.jetbrains.annotations.NotNull;
import org.junit.jupiter.api.AfterAll;
import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.firefox.FirefoxDriver;
import org.openqa.selenium.firefox.GeckoDriverService;
import pl.kielce.tu.pages.*;

import java.io.File;

import static org.junit.jupiter.api.Assertions.*;
import static org.junit.jupiter.api.Assumptions.assumeTrue;


class ShopTest {
        private static WebDriver driver;

        @BeforeAll
        static void setDriver() {
                //driver = new FirefoxDriver();
                driver = getFirefoxDriver();
                driver.manage().window().maximize();
        }

        @Test
        void getLoginPage() {
                final var expectedTitle = "Swag Labs";
                var loginPage = new ShopLoginPage(driver);
                loginPage.goToLoginPage();
                assertEquals(expectedTitle, loginPage.getTitle());
        }

        @Test
        void loginToShop() {
                final var expectedUrlSuffix = "/inventory.html";
                LogIntoApplication(driver);
                assertTrue(driver.getCurrentUrl().endsWith(expectedUrlSuffix));
                LogOutFromApplication(driver);
        }

        @Test
        void findProduct() {
                final var expected = "Sauce Labs Bike Light";
                LogIntoApplication(driver);
                var mainShopPage = new ShopMainPage(driver);
                var bikeLight = mainShopPage.findProduct(expected);
                assertEquals(expected, bikeLight.getText());
                LogOutFromApplication(driver);
        }

        @Test
        void checkProductDetails() {
                final var expectedName = "Sauce Labs Bike Light";
                final var expectedPrice = "$9.99";
                LogIntoApplication(driver);
                new ShopMainPage(driver)
                        .findProduct(expectedName)
                        .click();
                var product = new ProductPage(driver);
                assertAll("details",
                        ()->assertEquals(expectedName,product.getName()),
                        ()->assertEquals(expectedPrice,product.getPrice())
                );
                LogOutFromApplication(driver);
        }

        @Test
        void addBikeLightToCart() {
                final var productName = "Sauce Labs Bike Light";
                LogIntoApplication(driver);
                new ShopMainPage(driver)
                        .findProduct(productName)
                        .click();
                var product = new ProductPage(driver);
                product.addToCart();
                var cart = new CartIcon(driver);
                assertAll(
                        ()->assertTrue(cart.checkIfBadgePresent()),
                        ()->assertTrue(cart.getItemsNumber()>0)
                );
                LogOutFromApplication(driver);
        }

        @Test
        void addBackpackToCart() {
                final var productName = "Sauce Labs Backpack";
                LogIntoApplication(driver);
                new ShopMainPage(driver)
                        .findProduct(productName)
                        .click();
                var product = new ProductPage(driver);
                product.addToCart();
                var cartIcon = new CartIcon(driver);
                assertAll(
                        ()->assertTrue(cartIcon.checkIfBadgePresent()),
                        ()->assertTrue(cartIcon.getItemsNumber()>0)
                );
                LogOutFromApplication(driver);
        }

        @Test
        void placeOrder() {
                final var expectedMessage = "Thank you for your order!";
                LogIntoApplication(driver);
                var cartIcon = new CartIcon(driver);
                assumeTrue(cartIcon.getItemsNumber()>0);
                cartIcon.clickCart();
                var cartPage = new CartPage(driver);
                cartPage.clickCheckoutButton()
                        .fillForm("John","Doe","00-000")
                        .clickFinish();
                assertEquals(expectedMessage,cartPage.getFinalMessage());
                LogOutFromApplication(driver);
        }

        @Test
        void logOut() {
                final var expectedURL = "https://www.saucedemo.com/";
                LogIntoApplication(driver);
                LogOutFromApplication(driver);
                assertEquals(expectedURL, driver.getCurrentUrl());
        }

        @AfterAll
        static void tearDown() {
                if(driver!=null)
                        driver.quit();
        }

        private static FirefoxDriver getFirefoxDriver() {
                GeckoDriverService service = null;
                File snapGeckoDriver = new File("/snap/bin/geckodriver") {
                        // Workaround for the Firefox installed from snap based on:
                        // https://github.com/SeleniumHQ/selenium/issues/7788
                        @NotNull
                        public String getCanonicalPath() {
                                return getAbsolutePath();
                        }
                };
                if (snapGeckoDriver.exists()) {
                        service = new GeckoDriverService.Builder().usingDriverExecutable(snapGeckoDriver).build();
                }
                return service != null ? new FirefoxDriver(service) : new FirefoxDriver();
        }

        private void LogIntoApplication(WebDriver driver) {
                var loginPage = new ShopLoginPage(driver);
                loginPage.goToLoginPage();
                var userName = loginPage.getRandomUserName();
                var password = loginPage.getPassword();
                loginPage.enterUserName(userName);
                loginPage.enterPassword(password);
                loginPage.clickLogin();
        }

        private void LogOutFromApplication(WebDriver driver) {
                new Menu(driver)
                        .openMenu()
                        .clickLogout();
        }
}
